document.addEventListener('DOMContentLoaded', function() {
  // DOM Elements
  const cookieInput = document.getElementById('cookie-input');
  const addCookiesBtn = document.getElementById('add-cookies');
  const exportCookiesBtn = document.getElementById('export-cookies');
  const statusDiv = document.getElementById('status');
  const guideLink = document.getElementById('guide-link');

  // Cấu hình domain Shopee
  const SHOPEE_ROOT_DOMAIN = 'shopee.vn';
  const SHOPEE_DOMAIN = `.${SHOPEE_ROOT_DOMAIN}`;
  const SHOPEE_URL = `https://${SHOPEE_ROOT_DOMAIN}`;

  // Hướng dẫn sử dụng
  guideLink.href = "https://example.com/huong-dan-them-cookie-shopee";

  // Main function to add cookies
  addCookiesBtn.addEventListener('click', async () => {
    try {
      const input = cookieInput.value.trim();
      if (!input) throw new Error("Please enter cookies");

      // Parse cookies (support both string and JSON)
      const cookies = parseCookies(input);
      if (cookies.length === 0) throw new Error("No valid cookies found");

      // Add each cookie
      let successCount = 0;
      for (const cookie of cookies) {
        try {
          await setShopeeCookie(cookie);
          successCount++;
        } catch (err) {
          console.error(`Error adding ${cookie.name}:`, err);
        }
      }

      // Show result
      if (successCount > 0) {
        showStatus(`\u2705 Added ${successCount} cookies successfully!`, 'success');
        
        // Debug: Check cookies right after adding
        setTimeout(() => {
          chrome.cookies.getAll({ domain: SHOPEE_ROOT_DOMAIN }, (cookies) => {
            console.log('Current cookies:', cookies);
          });
        }, 500);
        
        // Auto refresh Shopee tabs
        refreshShopeeTabs();
      } else {
        throw new Error("No cookies could be added!");
      }
    } catch (error) {
      showStatus(`\u274c Error: ${error.message}`, 'error');
      console.error("Error details:", error);
    }
  });

  // Export cookies function
  exportCookiesBtn.addEventListener('click', async () => {
    try {
      const cookies = await getShopeeCookies();
      
      if (cookies.length === 0) {
        showStatus('No Shopee cookies found!', 'error');
        return;
      }
      
      // Prepare both formats
      const jsonOutput = JSON.stringify(cookies, null, 2);
      const stringOutput = cookies.map(c => 
        `${c.domain.replace(/^\./, '')}=${c.name}=${c.value}`
      ).join(';') + ';';
      
      cookieInput.value = `// JSON Format:\n${jsonOutput}\n\n// String Format:\n${stringOutput}`;
      showStatus(`\ud83d\udccb Exported ${cookies.length} cookies!`, 'success');
    } catch (error) {
      showStatus(`\u274c Error exporting cookies: ${error.message}`, 'error');
      console.error('Export error:', error);
    }
  });

  // ========== CÁC HÀM HỖ TRỢ ========== //

  // Parse cookie từ input
  function parseCookies(input) {
    // Xử lý JSON format
    if (input.startsWith('[')) {
      try {
        return JSON.parse(input).map(c => normalizeCookie(c));
      } catch (e) {
        throw new Error("JSON không hợp lệ: " + e.message);
      }
    }
    // Xử lý string format (hỗ trợ cả dạng domain=name=value và name=value)
    return input.split(';')
      .filter(pair => pair.includes('='))
      .map(pair => {
        const parts = pair.trim().split('=');
        if (parts.length >= 3) {
          // Định dạng: domain=name=value (value có thể chứa dấu '=')
          const domain = parts[0];
          const name = parts[1];
          const value = parts.slice(2).join('=');
          return normalizeCookie({
            name: name,
            value: value,
            domain: domain
          });
        } else if (parts.length === 2) {
          // Định dạng: name=value
          const [name, value] = parts;
          return normalizeCookie({
            name: name,
            value: value,
            domain: SHOPEE_ROOT_DOMAIN
          });
        } else {
          throw new Error('Cookie không đúng định dạng!');
        }
      });
  }

  // Chuẩn hóa cookie object
  function normalizeCookie(cookie) {
    return {
      name: cookie.name,
      value: cookie.value,
      domain: cookie.domain || SHOPEE_DOMAIN,
      path: cookie.path || '/',
      secure: cookie.secure !== false,
      sameSite: cookie.sameSite || 'lax',
      expirationDate: cookie.expirationDate || Math.floor(Date.now() / 1000) + 86400 // 1 ngày
    };
  }

  // Hàm set cookie với đầy đủ thuộc tính
  function setShopeeCookie(cookie) {
    return new Promise((resolve, reject) => {
      // Chuẩn hóa URL
      const url = cookie.domain.startsWith('.') 
        ? `https://${cookie.domain.substring(1)}` 
        : `https://${cookie.domain}`;

      const cookieDetails = {
        url: url,
        name: cookie.name,
        value: cookie.value,
        domain: cookie.domain,
        path: cookie.path,
        secure: cookie.secure,
        sameSite: cookie.sameSite,
        expirationDate: cookie.expirationDate
      };

      console.log('Setting cookie:', cookieDetails);
      
      chrome.cookies.set(cookieDetails, (details) => {
        if (chrome.runtime.lastError) {
          reject(chrome.runtime.lastError);
        } else {
          console.log('Đã set cookie:', details);
          resolve(details);
        }
      });
    });
  }

  // Lấy tất cả cookie Shopee
  function getShopeeCookies() {
    return new Promise((resolve) => {
      chrome.cookies.getAll({ domain: SHOPEE_ROOT_DOMAIN }, (cookies) => {
        resolve(cookies || []);
      });
    });
  }

  // Refresh tất cả tab Shopee
  function refreshShopeeTabs() {
    chrome.tabs.query({ url: `*://*.${SHOPEE_ROOT_DOMAIN}/*` }, (tabs) => {
      tabs.forEach(tab => {
        chrome.tabs.reload(tab.id);
        console.log('Đã refresh tab:', tab.url);
      });
    });
  }

  // Hiển thị thông báo
  function showStatus(message, type) {
    statusDiv.textContent = message;
    statusDiv.className = `status ${type}`;
    setTimeout(() => {
      statusDiv.textContent = '';
      statusDiv.className = 'status';
    }, 5000);
  }

  // Debug: Log cookies hiện có khi mở popup
  chrome.cookies.getAll({ domain: SHOPEE_ROOT_DOMAIN }, (cookies) => {
    console.log('Cookies ban đầu:', cookies);
  });
});